# Altiq AutoFill Extension - Installation & Testing Guide

End-to-end user guide:
- `docs/USER_GUIDE.md`

## 🚀 Quick Start

### Prerequisites

1. Backend API must be running on `http://localhost:8001`
2. Chrome or Chromium-based browser (Edge, Brave, etc.)
3. At least one document uploaded with extracted fields

### Step 1: Start the Backend

```bash
cd /opt/aura/altiq-autofill
source venv/bin/activate
uvicorn app.main:app --host 0.0.0.0 --port 8001 --reload
```

### Step 2: Load Extension in Chrome

1. Open Chrome and navigate to: `chrome://extensions/`
2. Enable **Developer mode** (toggle in top-right corner)
3. Click **"Load unpacked"**
4. Navigate to and select: `/opt/aura/altiq-autofill/extension`
5. The Altiq AutoFill icon should appear in your toolbar

### Step 3: Sign In

1. Click the Altiq AutoFill icon in your toolbar
2. Sign in

### Step 4: Upload Profiles/Documents

Upload profiles/documents in Admin Console: `altiq.io/admin` → **Autofill**

**Option B: Create Sample Fields Manually**

```bash
# Create sample user fields via API
curl -X POST http://localhost:8001/api/v1/fields/ \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "field_name": "Email",
    "field_type": "email",
    "value": "john.doe@example.com",
    "confidence": 0.95
  }'

curl -X POST http://localhost:8001/api/v1/fields/ \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "field_name": "First Name",
    "field_type": "firstName",
    "value": "John",
    "confidence": 0.90
  }'

curl -X POST http://localhost:8001/api/v1/fields/ \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "field_name": "Last Name",
    "field_type": "lastName",
    "value": "Doe",
    "confidence": 0.90
  }'

curl -X POST http://localhost:8001/api/v1/fields/ \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "field_name": "Phone",
    "field_type": "phone",
    "value": "(555) 123-4567",
    "confidence": 0.85
  }'
```

### Step 5: Test AutoFill

1. Open the test form: `file:///opt/aura/altiq-autofill/extension/test-form.html`
   - Or navigate to any website with a form (e.g., job application, contact form)

2. Click the Altiq AutoFill icon in your toolbar

3. Click **"AutoFill This Page"** button

4. Watch the magic happen! 🎉
   - Fields will be filled automatically
   - Border colors indicate confidence:
     - 🟢 **Green** = High confidence (≥80%)
     - 🟠 **Orange** = Medium confidence (50-79%)
     - 🔴 **Red** = Low confidence (<50%)

### Step 6: Test Learning Feature

1. After auto-fill, manually change a field value
2. Tab out or click away from the field
3. The border will turn green, indicating the correction was learned
4. Try auto-filling another form - your correction will be used

## 🧪 Testing Checklist

### Basic Functionality

- [ ] Extension loads without errors
- [ ] Can create an account
- [ ] Can sign in with existing account
- [ ] Dashboard shows document and field counts
- [ ] Can sign out and sign back in

### AutoFill Testing

- [ ] Detects form fields correctly
- [ ] Fills email field
- [ ] Fills name fields (first, last)
- [ ] Fills phone number
- [ ] Fills address fields
- [ ] Shows confidence colors correctly
- [ ] Notification appears after filling

### Learning & Corrections

- [ ] Can manually correct a filled field
- [ ] Border turns green after correction
- [ ] Correction is saved to backend
- [ ] Next auto-fill uses corrected value

### Edge Cases

- [ ] Works on forms without labels
- [ ] Handles fields with placeholders only
- [ ] Works with dynamic forms (JavaScript-generated)
- [ ] Doesn't fill hidden or disabled fields
- [ ] Doesn't interfere with form validation

## 🐛 Debugging

### View Console Logs

**Popup Console:**
- Right-click the extension icon
- Select "Inspect popup"
- Check Console tab

**Content Script Console:**
- Open DevTools on any webpage (F12)
- Check Console tab
- Look for "Altiq AutoFill" messages

**Background Service Worker:**
- Go to `chrome://extensions/`
- Find Altiq AutoFill
- Click "Inspect views: service worker"

### Common Issues

**Issue: "No fillable fields found"**
- Check if the page actually has input fields
- Try refreshing the page
- Check console for JavaScript errors

**Issue: "No stored data found"**
- Make sure you've uploaded and parsed documents
- Verify fields exist: `curl http://localhost:8001/api/v1/fields/ -H "Authorization: Bearer $TOKEN"`

**Issue: "Failed to get user info"**
- Token may be expired (30-minute expiry)
- Sign out and sign back in

**Issue: Extension not loading**
- Check for errors in `chrome://extensions/`
- Verify all files are present in the extension directory
- Try reloading the extension

**Issue: API connection failed**
- Verify backend is running on port 8001
- Check if CORS is configured correctly
- Look at Network tab in DevTools

## 📊 Testing with Real Data

### Generate Test Resume

Create a sample resume PDF or use this Python script:

```python
from reportlab.lib.pagesizes import letter
from reportlab.pdfgen import canvas

def create_test_resume():
    c = canvas.Canvas("/tmp/test_resume.pdf", pagesize=letter)
    
    c.setFont("Helvetica-Bold", 16)
    c.drawString(100, 750, "John Doe")
    
    c.setFont("Helvetica", 12)
    c.drawString(100, 730, "Email: john.doe@example.com")
    c.drawString(100, 715, "Phone: (555) 123-4567")
    c.drawString(100, 700, "Address: 123 Main St, San Francisco, CA 94102")
    c.drawString(100, 685, "LinkedIn: linkedin.com/in/johndoe")
    
    c.drawString(100, 650, "EXPERIENCE")
    c.drawString(100, 630, "Senior Software Engineer at Acme Corp")
    c.drawString(100, 615, "2020 - Present")
    
    c.save()
    print("Resume created: /tmp/test_resume.pdf")

create_test_resume()
```

Then upload and parse this resume through the API.

## 🔄 Development Workflow

1. Make changes to extension files
2. Go to `chrome://extensions/`
3. Click the **refresh icon** on Altiq AutoFill card
4. Test changes on a form
5. Check console for errors
6. Repeat

## 🎯 Performance Testing

### Measure AutoFill Speed

Open DevTools console on a test form and run:

```javascript
console.time('autofill');
// Click AutoFill button
// Wait for completion
console.timeEnd('autofill');
```

Typical times:
- Simple form (5-10 fields): 100-300ms
- Complex form (20+ fields): 300-800ms
- With vector search: +200-500ms per field

## 🚢 Production Deployment

### Update API Endpoint

Before deploying to production, update the API URL:

**In `popup.js`:**
```javascript
const API_BASE_URL = 'https://autofill.altiq.io/api/v1';
```

**In `content.js`:**
```javascript
const API_BASE_URL = 'https://autofill.altiq.io/api/v1';
```

**In `background.js`:**
```javascript
const API_BASE_URL = 'https://autofill.altiq.io/api/v1';
```

### Package for Chrome Web Store

```bash
cd /opt/aura/altiq-autofill
zip -r altiq-autofill-extension.zip extension/ -x "*.git*" "*/test-form.html" "*/README.md"
```

Upload `altiq-autofill-extension.zip` to Chrome Web Store Developer Dashboard.

## 📞 Support

For issues or questions:
- Check logs in DevTools
- Review API responses in Network tab
- Contact: support@altiq.io
