// Options page script - API configuration

const DEFAULT_PROD_URL = 'https://autofill.altiq.io/api/v1';
const DEFAULT_DEV_URL = 'https://dev-autofill.altiq.io/api/v1';
const DEFAULT_LOCAL_URL = 'http://localhost:8001/api/v1';

function normalizeApiBaseUrl(raw) {
    const url = new URL(String(raw || '').trim());
    // Require explicit scheme and /api/v1 prefix
    if (!['http:', 'https:'].includes(url.protocol)) {
        throw new Error('URL must be http(s)');
    }
    const path = url.pathname.replace(/\/+$/, '');
    if (!path.endsWith('/api/v1')) {
        throw new Error('URL must end with /api/v1');
    }
    // Normalize: strip trailing slash
    url.pathname = path;
    url.hash = '';
    return url.toString();
}

function isAllowedApiBaseUrl(normalizedUrl) {
    const url = new URL(normalizedUrl);

    // Allowlist by environment (industry standard: least trust for configurable endpoints)
    const allowed = new Set([
        DEFAULT_PROD_URL,
        DEFAULT_DEV_URL,
        DEFAULT_LOCAL_URL,
        'http://127.0.0.1:8001/api/v1',
        'http://[::1]:8001/api/v1'
    ]);

    // Also allow localhost loopback on arbitrary port for local dev
    const isLoopbackHost = ['localhost', '127.0.0.1', '::1', '[::1]'].includes(url.hostname);
    if (isLoopbackHost && url.protocol === 'http:') {
        return true;
    }

    return allowed.has(url.toString());
}

// DOM elements
const prodRadio = document.getElementById('prodServer');
const devRadio = document.getElementById('devServer');
const localRadio = document.getElementById('localServer');
const customUrlGroup = document.getElementById('customUrlGroup');
const apiUrlInput = document.getElementById('apiUrl');
const currentUrlInput = document.getElementById('currentUrl');
const saveBtn = document.getElementById('saveBtn');
const resetBtn = document.getElementById('resetBtn');
const messageDiv = document.getElementById('message');

// Load current settings
async function loadSettings() {
    const result = await chrome.storage.local.get(['api_server_type', 'api_base_url']);
    
    const serverType = result.api_server_type || 'production';
    const baseUrl = result.api_base_url || DEFAULT_PROD_URL;
    
    if (serverType === 'local') {
        localRadio.checked = true;
        devRadio.checked = false;
        prodRadio.checked = false;
        customUrlGroup.style.display = 'block';
        apiUrlInput.value = baseUrl;
    } else if (serverType === 'development') {
        devRadio.checked = true;
        localRadio.checked = false;
        prodRadio.checked = false;
        customUrlGroup.style.display = 'none';
    } else {
        prodRadio.checked = true;
        devRadio.checked = false;
        localRadio.checked = false;
        customUrlGroup.style.display = 'none';
    }
    
    currentUrlInput.value = baseUrl;
}

// Handle radio change
prodRadio.addEventListener('change', () => {
    customUrlGroup.style.display = 'none';
    currentUrlInput.value = DEFAULT_PROD_URL;
});

devRadio.addEventListener('change', () => {
    customUrlGroup.style.display = 'none';
    currentUrlInput.value = DEFAULT_DEV_URL;
});

localRadio.addEventListener('change', () => {
    customUrlGroup.style.display = 'block';
    apiUrlInput.value = apiUrlInput.value || DEFAULT_LOCAL_URL;
    currentUrlInput.value = apiUrlInput.value;
});

apiUrlInput.addEventListener('input', () => {
    if (localRadio.checked) {
        currentUrlInput.value = apiUrlInput.value;
    }
});

// Save settings
saveBtn.addEventListener('click', async () => {
    const serverType = prodRadio.checked
        ? 'production'
        : (devRadio.checked ? 'development' : 'local');
    const baseUrl = serverType === 'production'
        ? DEFAULT_PROD_URL
        : (serverType === 'development' ? DEFAULT_DEV_URL : apiUrlInput.value);
    
    // Validate/normalize URL
    let normalizedBaseUrl = baseUrl;
    if (serverType === 'local') {
        try {
            normalizedBaseUrl = normalizeApiBaseUrl(baseUrl);
        } catch (e) {
            showMessage('error', 'Invalid URL. Use http(s)://<host>:<port>/api/v1');
            return;
        }

        if (!isAllowedApiBaseUrl(normalizedBaseUrl)) {
            showMessage('error', 'For security, only loopback/local URLs are allowed in Local mode.');
            return;
        }
    }
    
    await chrome.storage.local.set({
        api_server_type: serverType,
        api_base_url: normalizedBaseUrl
    });
    
    currentUrlInput.value = normalizedBaseUrl;
    showMessage('success', '✅ Settings saved! Extension will use this URL for all API calls.');
    
    // Notify background script to reload config
    chrome.runtime.sendMessage({ type: 'RELOAD_CONFIG' });
});

// Reset to default
resetBtn.addEventListener('click', async () => {
    prodRadio.checked = true;
    devRadio.checked = false;
    localRadio.checked = false;
    customUrlGroup.style.display = 'none';
    apiUrlInput.value = '';
    currentUrlInput.value = DEFAULT_PROD_URL;
    
    await chrome.storage.local.set({
        api_server_type: 'production',
        api_base_url: DEFAULT_PROD_URL
    });
    
    showMessage('success', '✅ Reset to default production server.');
    
    chrome.runtime.sendMessage({ type: 'RELOAD_CONFIG' });
});

function showMessage(type, text) {
    messageDiv.className = `message ${type} active`;
    messageDiv.textContent = text;
    
    setTimeout(() => {
        messageDiv.classList.remove('active');
    }, 5000);
}

// Load on startup
loadSettings();
