#!/bin/bash
# Test helper script for Altiq AutoFill Extension

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(dirname "$SCRIPT_DIR")"

echo "🎯 Altiq AutoFill Extension - Test Helper"
echo "=========================================="
echo ""

# Check if backend is running
echo "📡 Checking backend status..."
if curl -s http://localhost:8001/docs > /dev/null 2>&1; then
    echo "✅ Backend is running on http://localhost:8001"
else
    echo "❌ Backend is NOT running"
    echo ""
    echo "Start the backend with:"
    echo "  cd $PROJECT_ROOT"
    echo "  source venv/bin/activate"
    echo "  uvicorn app.main:app --host 0.0.0.0 --port 8001 --reload"
    echo ""
    exit 1
fi

echo ""
echo "📦 Extension Location:"
echo "  $SCRIPT_DIR"
echo ""

echo "🔧 Installation Steps:"
echo "  1. Open Chrome: chrome://extensions/"
echo "  2. Enable 'Developer mode' (top-right toggle)"
echo "  3. Click 'Load unpacked'"
echo "  4. Select directory: $SCRIPT_DIR"
echo ""

echo "🧪 Test Form:"
echo "  file://$SCRIPT_DIR/test-form.html"
echo ""

# Offer to open test form
if command -v xdg-open > /dev/null 2>&1; then
    read -p "Open test form in browser? (y/n) " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        xdg-open "file://$SCRIPT_DIR/test-form.html"
        echo "✅ Test form opened"
    fi
elif command -v open > /dev/null 2>&1; then
    read -p "Open test form in browser? (y/n) " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        open "file://$SCRIPT_DIR/test-form.html"
        echo "✅ Test form opened"
    fi
fi

echo ""
echo "📚 Documentation:"
echo "  - README: $SCRIPT_DIR/README.md"
echo "  - Installation Guide: $SCRIPT_DIR/INSTALLATION.md"
echo ""

echo "✨ Quick Test:"
echo "  1. Load extension in Chrome"
echo "  2. Create account in extension popup"
echo "  3. Upload a document via API:"
echo ""
echo "     # Get token"
echo "     curl -X POST http://localhost:8001/api/v1/auth/login \\"
echo "       -H 'Content-Type: application/x-www-form-urlencoded' \\"
echo "       -d 'username=YOUR_EMAIL&password=YOUR_PASSWORD'"
echo ""
echo "     # Upload document"
echo "     curl -X POST http://localhost:8001/api/v1/documents/upload \\"
echo "       -H 'Authorization: Bearer YOUR_TOKEN' \\"
echo "       -F 'file=@/path/to/resume.pdf'"
echo ""
echo "     # Parse document"
echo "     curl -X POST http://localhost:8001/api/v1/documents/DOC_ID/parse \\"
echo "       -H 'Authorization: Bearer YOUR_TOKEN'"
echo ""
echo "  4. Open test form and click 'AutoFill This Page' in extension"
echo ""

echo "🎉 Happy Testing!"
